<?php

/*  PEL: PHP Exif Library.  A library with support for reading and
 *  writing all Exif headers in JPEG and TIFF images using PHP.
 *
 *  Copyright (C) 2004, 2005, 2006  Martin Geisler.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program in the file COPYING; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 *  Boston, MA 02110-1301 USA
 */

/* $Id$ */


/**
 * Classes used to hold longs, both signed and unsigned.
 *
 * @author Martin Geisler <mgeisler@users.sourceforge.net>
 * @version $Revision: 419 $
 * @date $Date: 2006-02-20 17:22:36 +0100 (Mon, 20 Feb 2006) $
 * @license http://www.gnu.org/licenses/gpl.html GNU General Public
 * License (GPL)
 * @package PEL
 */

/**#@+ Required class definitions. */
require_once('PelEntryNumber.php');
/**#@-*/


/**
 * Class for holding unsigned longs.
 *
 * This class can hold longs, either just a single long or an array of
 * longs.  The class will be used to manipulate any of the Exif tags
 * which can have format {@link PelFormat::LONG} like in this
 * example:
 * <code>
 * $w = $ifd->getEntry(PelTag::EXIF_IMAGE_WIDTH);
 * $w->setValue($w->getValue() / 2);
 * $h = $ifd->getEntry(PelTag::EXIF_IMAGE_HEIGHT);
 * $h->setValue($h->getValue() / 2);
 * </code>
 * Here the width and height is updated to 50% of their original
 * values.
 *
 * @author Martin Geisler <mgeisler@users.sourceforge.net>
 * @package PEL
 */
class PelEntryLong extends PelEntryNumber {

  /**
   * Make a new entry that can hold an unsigned long.
   *
   * The method accept its arguments in two forms: several integer
   * arguments or a single array argument.  The {@link getValue}
   * method will always return an array except for when a single
   * integer argument is given here, or when an array with just a
   * single integer is given.
   *
   * This means that one can conveniently use objects like this:
   * <code>
   * $a = new PelEntryLong(PelTag::EXIF_IMAGE_WIDTH, 123456);
   * $b = $a->getValue() - 654321;
   * </code>
   * where the call to {@link getValue} will return an integer instead
   * of an array with one integer element, which would then have to be
   * extracted.
   *
   * @param PelTag the tag which this entry represents.  This
   * should be one of the constants defined in {@link PelTag},
   * e.g., {@link PelTag::IMAGE_WIDTH}, or any other tag which can
   * have format {@link PelFormat::LONG}.
   *
   * @param int $value... the long(s) that this entry will
   * represent or an array of longs.  The argument passed must obey
   * the same rules as the argument to {@link setValue}, namely that
   * it should be within range of an unsigned long (32 bit), that is
   * between 0 and 4294967295 (inclusive).  If not, then a {@link
   * PelExifOverflowException} will be thrown.
   */
  function __construct($tag /* $value... */) {
    $this->tag    = $tag;
    $this->min    = 0;
    $this->max    = 4294967295;
    $this->format = PelFormat::LONG;

    $value = func_get_args();
    array_shift($value);
    $this->setValueArray($value);
  }


  /**
   * Convert a number into bytes.
   *
   * @param int the number that should be converted.
   *
   * @param PelByteOrder one of {@link PelConvert::LITTLE_ENDIAN} and
   * {@link PelConvert::BIG_ENDIAN}, specifying the target byte order.
   *
   * @return string bytes representing the number given.
   */
  function numberToBytes($number, $order) {
    return PelConvert::longToBytes($number, $order);
  }
}


/**
 * Class for holding signed longs.
 *
 * This class can hold longs, either just a single long or an array of
 * longs.  The class will be used to manipulate any of the Exif tags
 * which can have format {@link PelFormat::SLONG}.
 *
 * @author Martin Geisler <mgeisler@users.sourceforge.net>
 * @package PEL
 */
class PelEntrySLong extends PelEntryNumber {

  /**
   * Make a new entry that can hold a signed long.
   *
   * The method accept its arguments in two forms: several integer
   * arguments or a single array argument.  The {@link getValue}
   * method will always return an array except for when a single
   * integer argument is given here, or when an array with just a
   * single integer is given.
   *
   * @param PelTag the tag which this entry represents.  This
   * should be one of the constants defined in {@link PelTag}
   * which have format {@link PelFormat::SLONG}.
   *
   * @param int $value... the long(s) that this entry will represent
   * or an array of longs.  The argument passed must obey the same
   * rules as the argument to {@link setValue}, namely that it should
   * be within range of a signed long (32 bit), that is between
   * -2147483648 and 2147483647 (inclusive).  If not, then a {@link
   * PelOverflowException} will be thrown.
   */
  function __construct($tag /* $value... */) {
    $this->tag    = $tag;
    $this->min    = -2147483648;
    $this->max    = 2147483647;
    $this->format = PelFormat::SLONG;

    $value = func_get_args();
    array_shift($value);
    $this->setValueArray($value);
  }


  /**
   * Convert a number into bytes.
   *
   * @param int the number that should be converted.
   *
   * @param PelByteOrder one of {@link PelConvert::LITTLE_ENDIAN} and
   * {@link PelConvert::BIG_ENDIAN}, specifying the target byte order.
   *
   * @return string bytes representing the number given.
   */
  function numberToBytes($number, $order) {
    return PelConvert::sLongToBytes($number, $order);
  }
}


?>